﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.Threading;

/* A sample to emphasize on the necessity of thread synchronization, 
   when multiple threads access the same resource,
   which necessity is caused by the parallel execution of those multiple threads. 
   
   This sample has an ArrayList containing a large number of double values.
   That list is accessed by both, the primary thread and a secondary thread.
 
   The primary thread first alters list values and then sums the total.
   The secondary thread just sums the total.
 
   As you can see the total of the one is never the same as the total of the other 
   although both threads sum the same list.
 
   This incosistency happens because a common resource, the list in this case,
   is not protected.
 */
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
            synContext = WindowsFormsSynchronizationContext.Current;

            for (int i = 0; i < 1000000; i++)       // initialize list values
                list.Add((double)i);
        }

        SynchronizationContext synContext;
        ArrayList list = new ArrayList();
        Thread t = null;


        /* a helper class just to convey info through the SynchronizationContext.Send() method */
        public class ExecInfo
        {
            int threadID;
            double total;

            public ExecInfo(int ThreadID, double Total)
            {
                this.threadID = ThreadID;
                this.total = Total;
            }

            public override string ToString()
            {
                return "Thread ID: " + threadID.ToString() + ", total: " + total.ToString();
            }
        }

        /* the thread procedure */
        void ThreadProc()
        {
            double total = 0;

            for (int i = 0; i < list.Count; i++)
                total += (double)list[i];

            /* each thread is assigned a unique ID by the system.
               The Thread.CurrentThread static property returns the currently executed thread
               while the Thread.ManagedThreadId instance property returns that unique ID */
            synContext.Send(SynchronizedMethod, new ExecInfo(Thread.CurrentThread.ManagedThreadId, total));
        }

        void SynchronizedMethod(object state)
        {
            textBox1.Text += state.ToString() + Environment.NewLine;
        }

        private void button1_Click(object sender, EventArgs e)
        {
            if ((t == null) || (!t.IsAlive))     
            {
                t = new Thread(ThreadProc);     
                t.Start();

                /* alter list values
                   at the same time, the thread t tries to sum the list */
                for (int i = 0; i < list.Count; i++)
                    list[i] = ((double)list[i]) / 2.0;

                /* a method is executed in a thread context 
                   ThreadProc() is executed twice by this application, and in two different thread contexts: 
                   once for the primary thread and once for the t thread */
                ThreadProc();
            }
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = (t != null) && (t.IsAlive);

            if (e.Cancel)
                MessageBox.Show("Please wait! A thread is still executed...");
        }
    }
}
